% Copyright (c) 2013, Massachusetts Institute of Technology
% This program was presented in the book "Visual Psychophysics:
% From Laboratory to Theory" by Zhong-Lin Lu and Barbara Dosher.
% The book is available at http://mitpress.mit.edu/books/visual-psychophysics

%%% Program ContrastSensitivityFunction.m

function ContrastSensitivityFunction

%% Display Setup Module
% Define display parameters
whichScreen = max(Screen('screens'));
p.ScreenDistance = 30;  % in inches
p.ScreenHeight = 15;    % in inches
p.ScreenGamma = 2;  % from monitor calibration
p.maxLuminance = 100; % from monitor calibration
p.ScreenBackground = 0.5; 

% Open the display window, set up lookup table, and hide the 
% mouse cursor
if exist('onCleanup', 'class'), oC_Obj = onCleanup(@()sca); end  
        % close any pre-existing PTB Screen window
% Prepare setup of imaging pipeline for onscreen window. 
PsychImaging('PrepareConfiguration'); % First step in starting
                                      % pipeline
PsychImaging('AddTask', 'General',  'FloatingPoint32BitIfPossible');  
        % set up a 32-bit floatingpoint framebuffer
PsychImaging('AddTask', 'General', 'NormalizedHighresColorRange');   
        % normalize the color range ([0, 1] corresponds 
        % to [min, max])
PsychImaging('AddTask', 'General', 'EnablePseudoGrayOutput'); 
        % enable high gray level resolution output with 
        % bitstealing
PsychImaging('AddTask','FinalFormatting',  'DisplayColorCorrection','SimpleGamma');  
        % setup Gamma correction method using simple power 
        % function for all color channels 
[windowPtr p.ScreenRect] = PsychImaging('OpenWindow', whichScreen, p.ScreenBackground);  
        % Finishes the setup phase for imaging pipeline
        % creates an onscreen window, performs all remaining  
        % configuration steps
PsychColorCorrection('SetEncodingGamma', windowPtr, 1/ p.ScreenGamma);  
% set Gamma for all color channels
HideCursor;  % Hide the mouse cursor 

% Get frame rate and set screen font
p.ScreenFrameRate = FrameRate(windowPtr); 
        % get current frame rate
Screen('TextFont', windowPtr, 'Times'); 
        % set the font for the screen to Times
Screen('TextSize', windowPtr, 24); % set the font size 
                                   % for the screen to 24
                                   
%% Experimental Module

% specify the experimental parameters
nContrast = 7; % number of contrast levels
nSF = 9;       % number of spatial frequencies 
repeats = 10;  % number of trials to repeat for each condition
nTrials = repeats * nContrast * nSF; 
               % total number of trials
keys = {'1' '2' 'esc'};   % keys to respond for which 
                          % interval, and to break
p.randSeed = ClockRandSeed;     
         % use clock to set random number generator

% Specify the stimulus
p.stimSize = 6;           % image diameter in visual degree
conRange = [0.0025 0.04]; % lowest and highest tested contrast
sfRange = [0.125 32];     % lowest and highest tested sf in 
                          % cycles/deg
p.stimDuration = 0.1;     % stimulus duration in seconds
p.interval = 1; % seconds between two intervals of a trial
p.ISI = 0.5;              % seconds between trials

% Compute stimulus parameters
ppd = pi/180 * p.ScreenDistance / p.ScreenHeight * ...
       p.ScreenRect(4);  % compute pixels per degree from 
                         % p.ScreenDistance andp.ScreenHeight
m = round(p.stimSize * ppd /2) * 2; % stimulus size in pixels
fixRect = CenterRect([0 0 1 1] * 8, p.ScreenRect);  
        % define an 8 x 8 pixel fixation 
       
% create another fixation stimulus of horizontal and vertical cross hairs outside the area of the sine wave pattern
fixLen = 32; % length of fixation in pixels
fixXY = [ [-1 0 0 1]*fixLen + [-1 -1 1 1 ]*m/2 + ...
        p.ScreenRect(3)/2 [1 1 1 1] * p.ScreenRect(3)/2; ...
        [1 1 1 1]*p.ScreenRect(4)/2 [-1 0 0 1] *fixLen + ...
        [-1 -1 1 1 ] * m / 2 + p.ScreenRect(4) / 2];  

p.contrasts = logspace(log10(conRange(1)), ...
              log10(conRange(2)), nContrast);
p.SFs = logspace(log10(sfRange(1)), log10(sfRange(2)), nSF);
SFs = p.SFs / ppd; 
        % compute spatial frequency in cycles per pixel

% Initialize a table to set up experimental conditions
p.recLabel = {'trialIndex' 'contrastIndex' 'SFIndex' ...
   'whichInterval' 'respCorrect' 'respTime'};    
        % labels the columns of the data recording array rec
rec = nan(nTrials, length(p.recLabel));         
        % matrix rec is initialized as an nTrials x 6 
        % matrix of NaN
rec(:, 1) = 1 : nTrials;  
        % initialize trial numbers from 1 to nTrials
contrastIndex = repmat(1 : nContrast, [nSF 1 repeats]);
SFIndex = repmat((1 : nSF)', [1 nContrast repeats]);
intervalIndex = ones(nSF, nContrast, repeats) * 2; 
        % first set all trials to 2nd interval
intervalIndex(:, :, 1 : repeats / 2) = 1; 
        % change fisrt half to 1
[rec(:, 2) ind] = Shuffle(contrastIndex(:)); 
        % shuffle contrast indexes to randomize
rec(:, 3) = SFIndex(ind); % shuffle SF indexes to randomize
rec(:, 4) = intervalIndex(ind);  
        % shuffle intervals to randomize
 
% Prioritize display to optimize display timing
Priority(MaxPriority(windowPtr));

% Start experiment with instructions
str = sprintf('Press 1 or 2 for first or second  interval.\n\nPress SPACE to start.');
DrawFormattedText(windowPtr, str, 'center', 'center', 1);
        % Draw instruction text string centered in window
Screen('Flip', windowPtr);  
        % flip the text image into active buffer
Beeper;
WaitTill('space');        % wait for space bar 
Screen('FillOval', windowPtr, 0, fixRect);   
        % create a black fixation box
Secs = Screen('Flip', windowPtr);   
        % flip the fixation image into active buffer
 
% procedural sinewavegrating allows us to change its 
% parameters very quickly
tex = CreateProceduralSineGrating(windowPtr, m, m, ...
      [1 1 1 0] * 0.5, m/2, 0.5);
        % "CreateProceduralSineGrating" is a psychtoolbox 
        % function to create a procedural texture for drawing 
        % sine grating stimulus patches on the GPU
 
% Run nTrials trials
for i = 1 : nTrials
    con = p.contrasts(rec(i, 2)); 
        % use contrast index from rec to set contrast 
        % for this trial
    sf = SFs(rec(i, 3)); 
        % use spatial frequency index from rec to set spatial
        % frequency for this trial
    flipSecs = Secs + p.ISI + [0 p.interval]; 
        % define the start time of the two intervals	
    
    for j = 1 : 2
        if rec(i, 4) == j  % draw the grating in the interval
                           % defined in rec(i,4) only
            Screen('DrawTexture', windowPtr, tex, [], [], ...
                  0, [], [], [], [], [], [180 sf con 0]);
        end   % draw the sine grating with phase 180, 
              % spatial frequency, and contrast
        Screen('DrawLines', windowPtr, fixXY, 3, 0); 
              % add the fixation crosshairs
        t0 = Screen('Flip', windowPtr, flipSecs(j)); 
              % show the stimulus and return the time 
        Screen('Flip', windowPtr, t0 + p.stimDuration); 
              % turn off the stimulus by flipping to 
              % background image after p.stimDuration secs
    end
        
    Screen('FillOval', windowPtr, 0, fixRect); 
       % draw the smaller centered fixation
    Screen('Flip', windowPtr);
        
    [key Secs] = WaitTill(keys);   
       % wait for response, return key and response time
    if iscellstr(key), key = key{1}; end 
       % take the first response in case of multiple key 
       % presses
    if strcmp(key, 'esc'), break; end 
       % check if response is <escape> to stop experiment
    rec(i, 5) = str2double(key) == rec(i, 4); 
       % record correctness
    rec(i, 6) = Secs - t0; % record respTime
    if rec(i, 5), Beeper; end % beep if correct
    
    Screen('FillOval', windowPtr, 0, fixRect); % draw fixation
    Screen('Flip', windowPtr);
end
 
save ContrastSensitivityFunction_rst rec p; % save the results



%% System Reinstatement Module
Priority(0);  % restore priority
sca; % close window and textures, restore color lookup table

